<?php

namespace App\Http\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Registro;
use App\Models\User;
use App\Models\Ugele;
use App\Models\Oficina;
use App\Models\Servicio;
use Carbon\Carbon;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;


class Tickets extends Component
{
    use WithPagination;
    use WithFileUploads;

    protected $paginationTheme = 'bootstrap';
    public $selected_id, $keyWord, $Cod_Registro, $Cod_Ugel, $Cod_Oficina, $Usuario, $Descripcion_problema, $Tecnico_asignado, $Estado, $Solucion, $Imagen, $imagenInput, $Fecha_Inicio, $Fecha_Final, $Servicio, $Prioridad, $Tipo_Soporte;

    protected $messages = [
        'Cod_Registro.required' => 'El campo Cod Registro es requerido.',
        'Cod_Ugel.required' => 'El campo Cod Ugel es requerido.',
        'Cod_Oficina.required' => 'El campo Cod Oficina es requerido.',
        'Usuario.required' => 'El campo Cod Usuario es requerido.',
        'Descripcion_problema.required' => 'El campo Decripcion de Problema es requerido.',
        'Servicio.required' => 'El campo Servicio es requerido.',
        'Tipo_Soporte.required' => 'El campo Tipo de Soporte es requerido.',
        'imagenInput.image' => 'El campo Imagen debe ser una imagen.',
        'imagenInput.mimes' => 'El campo Imagen debe ser de tipo jpeg,png,jpg o gif.',
        'imagenInput.max' => 'El campo Imagen debe ser de tamaño 500KB maximos.',
        'Fecha_Inicio.required' => 'El campo Fecha Inicio es requerido',
    ];
    
    public function render()
    {
        $grupougel = Oficina::where('Cod_Oficina', '=', strval(Auth::user()->Cod_Oficina))->first()->Cod_Ugel;
        $cod_usuario = Auth::user()->DNI;
        #servicios
        $servicios_hardware = Servicio::when(Auth::user()->Tipo === 'SuperAdministrador', function ($query) {
            $query->where('Cod_Ugel', $this->Cod_Ugel);
        })
            ->when(Auth::user()->Tipo !== 'SuperAdministrador', function ($query) use ($grupougel) {
                $query->where('Cod_Ugel', $grupougel);
            })
            ->where('Tipo', 'Hardware') // Filtro para servicios de tipo Hardware
            ->get();

        $servicios_software = Servicio::when(Auth::user()->Tipo === 'SuperAdministrador', function ($query) {
            $query->where('Cod_Ugel', $this->Cod_Ugel);
        })
            ->when(Auth::user()->Tipo !== 'SuperAdministrador', function ($query) use ($grupougel) {
                $query->where('Cod_Ugel', $grupougel);
            })
            ->where('Tipo', 'Software') // Filtro para servicios de tipo Software
            ->get();
        #usuarios
        $usuarios = User::when(Auth::user()->Tipo === 'Administrador', function ($query) use ($grupougel) {
            $query->whereHas('oficina', function ($query) use ($grupougel) {
                $query->where('Cod_Ugel', $grupougel);
            });
        })
            ->when(Auth::user()->Tipo === 'SuperAdministrador', function ($query) {
                $query->whereHas('oficina', function ($query) {
                    $query->where('Cod_Ugel', $this->Cod_Ugel);
                });
            })
            ->where('Tipo','!=', 'SuperAdministrador')
            ->where('Cod_Oficina', '=', strval($this->Cod_Oficina))
            ->get();
            
        $tecnicos = User::when(Auth::user()->Tipo === 'Administrador', function ($query) use ($grupougel) {
            $query->whereHas('oficina', function ($query) use ($grupougel) {
                $query->where('Cod_Ugel', $grupougel);
            });
        })
            ->when(Auth::user()->Tipo === 'SuperAdministrador', function ($query) {
                $query->whereHas('oficina', function ($query) {
                    $query->where('Cod_Ugel', $this->Cod_Ugel);
                });
            })
            ->where('Tipo', 'Tecnico')
            ->get();
        $ugeles = Ugele::all();
        $oficinas = Oficina::when(Auth::user()->Tipo === 'SuperAdministrador', function ($query) {
            $query->where('Cod_Ugel', $this->Cod_Ugel);
        })
            ->when(Auth::user()->Tipo === 'Administrador', function ($query) use ($grupougel) {
                $query->where('Cod_Ugel', $grupougel);
            })
            ->get();
        $keyWord = '%' . $this->keyWord . '%';
        #obtenemos una lista delos registros y les ponemos una enumeration la cual despues usaemos una vez filtrado
        #nuestros registros
        $ordentickets = Registro::select(
            'Cod_Registro',
            'Cod_Ugel',
            DB::raw('@row_number := CASE WHEN @current_u = Cod_Ugel THEN @row_number + 1 ELSE 0 END as enumeration'),
            DB::raw('@current_u := Cod_Ugel as dummy')
        )
            ->crossJoin(DB::raw('(SELECT @row_number := -1, @current_u := "") AS counter'))
            ->orderBy('Cod_Ugel')
            ->orderByRaw('Estado ASC, IF(Prioridad IS NULL, 0, 1) ASC, Prioridad DESC, Fecha_Inicio ASC')
            ->get();
                 
        
        #registros
        $registros = Registro::select('registros.*', 'Nombre_Oficina', 'Nombre_Ugel', 'users.Nombres_y_Apellidos as Nombres_y_Apellidos_Usuario', 'Nombre', 'u_tecnico.Nombres_y_Apellidos as Nombres_y_Apellidos_Tecnico')
            #unimos con usuer paramostrar el nombre
            ->leftJoin('users', 'registros.Usuario', '=', 'users.DNI')
            #unimos con ugel para mostrar la Oficina
            ->leftJoin('oficinas', 'oficinas.Cod_Oficina', '=', 'registros.Cod_Oficina')
            #unimos con ugel para mostrar la Ugel
            ->leftJoin('ugeles', 'ugeles.Cod_Ugel', '=', 'registros.Cod_Ugel')
            #unimos con usuer para mostrar el tecnico
            ->leftJoin('users as u_tecnico', 'registros.Tecnico_asignado', '=', 'u_tecnico.DNI')
            #unimos con seervicios para mostrar el servicio
            ->leftJoin('servicios', 'registros.Servicio', '=', 'servicios.Cod_Servicio')
            #filtramos todas los registros que le corresponden a la ugel del administrador
            ->when(Auth::user()->Tipo === 'Administrador', function ($query) use ($grupougel) {
                $query->where('registros.Cod_Ugel', $grupougel);
            })
            #filtramos todos los registros que le corresponden al usuario
            ->when(Auth::user()->Tipo === 'Tecnico', function ($query) use ($cod_usuario) {
                $query->where('registros.Tecnico_asignado', $cod_usuario);
            })
            ->when(Auth::user()->Tipo === 'Usuario', function ($query) use ($cod_usuario) {
                $query->where('registros.Usuario', $cod_usuario);
            })
            ->where('Estado', '!=', 'Terminado')
            ->where(function ($query) use ($keyWord) {
                $query->where('Cod_Registro', 'LIKE', $keyWord)
                    ->orWhere('Nombre_Ugel', 'LIKE', $keyWord)
                    ->orWhere('Nombre_Oficina', 'LIKE', $keyWord)
                    ->orWhere('Usuario', 'LIKE', $keyWord)
                    ->orWhere('Descripcion_problema', 'LIKE', $keyWord)
                    ->orWhere('Tecnico_asignado', 'LIKE', $keyWord)
                    ->orWhere('Estado', 'LIKE', $keyWord)
                    ->orWhere('Solucion', 'LIKE', $keyWord)
                    ->orWhere('Fecha_Inicio', 'LIKE', $keyWord)
                    ->orWhere('Fecha_Final', 'LIKE', $keyWord);
            })
            ->get();

        // Combinar los resultados de $ordentickets y $registros
        $combinedResults = [];

        foreach ($registros as $registro) {
            $ordenticket = $ordentickets->where('Cod_Registro', $registro->Cod_Registro)->first();

            if ($ordenticket) {
                $registro->enumeration = $ordenticket->enumeration;
                $combinedResults[] = $registro;
            }
        }
        // Ordenar $combinedResults por la propiedad enumeration
        $combinedResults = collect($combinedResults)->sortBy('enumeration')->values();

        // Formatear las fechas antes de pasar los resultados a la vista
        foreach ($combinedResults as $registro) {
            $registro->Fecha_Inicio = $registro->Fecha_Inicio ? Carbon::parse($registro->Fecha_Inicio)->format('d/m/y H:i:s') : '';
            $registro->Fecha_Final = $registro->Fecha_Final ? Carbon::parse($registro->Fecha_Final)->format('d/m/y H:i:s') : '';
        }

        return view('livewire.tickets.view', [
            'registros' => $combinedResults,
            'ugeles' => $ugeles,
            'oficinas' => $oficinas,
            'usuarios' => $usuarios,
            'tecnicos' => $tecnicos,
            'servicios_hardware' => $servicios_hardware,
            'servicios_software' => $servicios_software,
        ]);
    }

    public function cancel()
    {
        $this->resetInput();
    }

    private function resetInput()
    {
        $this->Cod_Registro = null;
        $this->Cod_Ugel = null;
        $this->Cod_Oficina = null;
        $this->Usuario = null;
        $this->Descripcion_problema = null;
        $this->Imagen = null;
        $this->imagenInput = null;
        $this->Servicio = null;
        $this->Tecnico_asignado = null;
        $this->Estado = null;
        $this->Prioridad = null;
        $this->Tipo_Soporte = null;
        $this->Solucion = null;
        $this->Fecha_Inicio = null;
        $this->Fecha_Final = null;
    }

    public function store()
    {
        $this->validate([
            'Descripcion_problema' => 'required',
            'Imagen' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        #capturamos la imagen y lo transformamos a base64
        if ($this->imagenInput instanceof \Illuminate\Http\UploadedFile) {
            $this->Imagen = base64_encode(file_get_contents($this->imagenInput->getRealPath()));
        }
        // Obtener la fecha y hora actual
        $this->Fecha_Inicio = date('Y-m-d H:i:s');

        // Inicializamos el Estado
        $this->Estado = 'Recibido';

        if (Auth::user()->Tipo == 'Usuario') {
            //Obtener el codigo de oficina
            $this->Cod_Oficina = Auth::user()->Cod_Oficina;
            //Obtener el codigo de Ugel
            $this->Cod_Ugel = Oficina::where('Cod_Oficina', Auth::user()->Cod_Oficina)->first()->Cod_Ugel;
            //Obtener el usuario
            $this->Usuario = Auth::user()->DNI;
        }
        if (Auth::user()->Tipo == 'Administrador') {
            //Obtener el codigo de Ugel
            if ($this->Cod_Ugel == "" or $this->Cod_Ugel == null) {
                $this->Cod_Ugel = Oficina::where('Cod_Oficina', Auth::user()->Cod_Oficina)->first()->Cod_Ugel;
            }
        }

        //Creac Cod de Registro
        $this->Cod_Registro = $this->Usuario . date('dmYHis');
        Registro::create([
            'Cod_Registro' => $this->Cod_Registro,
            'Cod_Ugel' => $this->Cod_Ugel,
            'Cod_Oficina' => $this->Cod_Oficina,
            'Usuario' => $this->Usuario,
            'Descripcion_problema' => $this->Descripcion_problema,
            'Imagen' => $this->Imagen,
            'Servicio' => $this->Servicio,
            'Tecnico_asignado' => $this->Tecnico_asignado,
            'Estado' => $this->Estado,
            'Prioridad' => $this->Prioridad,
            'Tipo_Soporte' => $this->Tipo_Soporte,
            'Solucion' => $this->Solucion,
            'Fecha_Inicio' => $this->Fecha_Inicio,
            'Fecha_Final' => $this->Fecha_Final,
        ]);

        $this->resetInput();
        $this->dispatchBrowserEvent('closeModal');
        session()->flash('message', 'Registro Creado Exitosamente.');
    }

    public function edit($id)
    {
        $record = Registro::findOrFail($id);
        $this->selected_id = $id;
        $this->Cod_Registro = $record->Cod_Registro;
        $this->Cod_Ugel = $record->Cod_Ugel;
        $this->Cod_Oficina = $record->Cod_Oficina;
        $this->Usuario = $record->Usuario;
        $this->Descripcion_problema = $record->Descripcion_problema;
        $this->Servicio = $record->Servicio;
        $this->Imagen = $record->Imagen;
        $this->Tecnico_asignado = $record->Tecnico_asignado;
        $this->Estado = $record->Estado;
        $this->Prioridad = $record->Prioridad;
        $this->Tipo_Soporte = $record->Tipo_Soporte;
        $this->Solucion = $record->Solucion;
        $this->Fecha_Inicio = $record->Fecha_Inicio;
        $this->Fecha_Final = $record->Fecha_Final;
    }

    public function update()
    {
        if(Auth::user()->Tipo==="SuperAdministrador"){
            $this->validate([
                'Cod_Registro' => 'required',
                'Cod_Ugel' => 'required',
                'Cod_Oficina' => 'required',
                'Usuario' => 'required',
                'Descripcion_problema' => 'required',
                'Servicio'=>'required',
                'Tipo_Soporte'=>'required',
                'Fecha_Inicio' => 'required',
                'imagenInput' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:1024',
            ]);
        }
        if(Auth::user()->Tipo==="Usuario"){
            $this->validate([
                'Descripcion_problema' => 'required',
                'Servicio'=>'required',
                'Tipo_Soporte'=>'required',
                'imagenInput' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:1024',
            ]);
        }

        if (
            (Auth::user()->Tipo === 'Administrador')
            && !($this->Tecnico_asignado === null || $this->Tecnico_asignado === "")
        ) {
            $this->Estado = "Proceso";
        }
        if (
            (Auth::user()->Tipo === 'Tecnico')
            && !($this->Solucion === null || $this->Solucion === "")
        ) {
            $this->Estado = "Terminado";
            $this->Fecha_Final = date('Y-m-d H:i:s');
        }
        if ($this->Tecnico_asignado === "" or $this->Tecnico_asignado === null) {
            $this->Tecnico_asignado = null;
        }
        if ($this->Fecha_Final === "") {
            $this->Fecha_Final = null;
        }
        if ($this->imagenInput instanceof \Illuminate\Http\UploadedFile) {
            $this->Imagen = base64_encode(file_get_contents($this->imagenInput->getRealPath()));
        }

        if ($this->selected_id) {
            if (Auth::user()->Tipo === 'SuperAdministrador') {
                $record = Registro::find($this->selected_id);
                $record->update([
                    'Cod_Registro' => $this->Cod_Registro,
                    'Cod_Ugel' => $this->Cod_Ugel,
                    'Cod_Oficina' => $this->Cod_Oficina,
                    'Usuario' => $this->Usuario,
                    'Descripcion_problema' => $this->Descripcion_problema,
                    'Servicio' => $this->Servicio,
                    'Tecnico_asignado' => $this->Tecnico_asignado,
                    'Estado' => $this->Estado,
                    'Prioridad' => $this->Prioridad,
                    'Tipo_Soporte' => $this->Tipo_Soporte,
                    'Solucion' => $this->Solucion,
                    'Fecha_Inicio' => $this->Fecha_Inicio,
                    'Fecha_Final' => $this->Fecha_Final,
                ]);

                // Actualizar la imagen solo si se proporciona una nueva
                if ($this->imagenInput instanceof \Illuminate\Http\UploadedFile) {
                    $record->update([
                        'Imagen' => $this->Imagen,
                    ]);
                }
                ;
            }
            if (Auth::user()->Tipo === 'Administrador') {
                $record = Registro::find($this->selected_id);
                $record->update([
                    'Prioridad' => $this->Prioridad,
                    'Tecnico_asignado' => $this->Tecnico_asignado,
                    'Estado' => $this->Estado,
                ]);
            }
            if (Auth::user()->Tipo === 'Tecnico') {
                $record = Registro::find($this->selected_id);
                $record->update([
                    'Estado' => $this->Estado,
                    'Solucion' => $this->Solucion,
                    'Fecha_Final' => $this->Fecha_Final,
                ]);
            }
            ;
            $this->resetInput();
            $this->dispatchBrowserEvent('closeModal');
            session()->flash('message', 'Registro Actualizado Exitosamente.');
        }
    }
}